/* =============================================================================
 *
 * Stats_Demo.c
 *
 * This file contains the code for the stats demo task plus the global
 * declarations of the task's TCB and stack.
 *
 * v1.0	YFS	Initial release. 
 * v1.1 YFS Updated for PSoC Creator 3.0. No changes in this file.
 *
 * =============================================================================
*/

/*
* Copyright (2013), Cypress Semiconductor Corporation. All Rights Reserved.
*
* This software is owned by Cypress Semiconductor Corporation (Cypress)
* and is protected by and subject to worldwide patent protection (United
* States and foreign), United States copyright laws and international treaty
* provisions. Cypress hereby grants to licensee a personal, non-exclusive,
* non-transferable license to copy, use, modify, create derivative works of,
* and compile the Cypress Source Code and derivative works for the sole
* purpose of creating custom software in support of licensee product to be
* used only in conjunction with a Cypress integrated circuit as specified in
* the applicable agreement. Any reproduction, modification, translation,
* compilation, or representation of this software except as specified above 
* is prohibited without the express written permission of Cypress.
*
* Disclaimer: CYPRESS MAKES NO WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, WITH 
* REGARD TO THIS MATERIAL, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
* Cypress reserves the right to make changes without further notice to the 
* materials described herein. Cypress does not assume any liability arising out 
* of the application or use of any product or circuit described herein. Cypress 
* does not authorize its products for use as critical components in life-support 
* systems where a malfunction or failure may reasonably be expected to result in 
* significant injury to the user. The inclusion of Cypress' product in a life-
* support systems application implies that the manufacturer assumes all risk of 
* such use and in doing so indemnifies Cypress against all charges. 
*
* Use of this Software may be limited by and subject to the applicable Cypress
* software license agreement. 
*/

/* Include Micrium APIs and defines */
#include <includes.h>

/* Include PSoC system and component APIs and defines */
#include <project.h>

/* Include application function declarations and defines */
#include <utils.h>
#include <LCD_Defs.h>
#include <Task_Defs.h>
#include <Capsense_Monitor_Q.h>

#define NUM_STATS (6)

#define STATS_VERSION (0)
#define STATS_TICK_RATE (1)
#define STATS_CPU_USAGE (2)
#define STATS_CPU_FREQ (3)
#define STATS_TICKS (4)
#define STATS_CTXSW (5)

/* Delay 100ms - reaonable update rate for human eye */
#define INFO_REFRESH_RATE (100)

/* Time to spend on each stat - 3s */
#define SCREEN_REFRESH_RATE (3000)

/* Number of for-loops to execute for each stat to hit above timimg */
#define TIME_TO_CHANGE (SCREEN_REFRESH_RATE / INFO_REFRESH_RATE )

/* Declare the Task Control Block and stack for Stats_Demo_Task */
OS_TCB	Stats_Demo_Task_TCB;
CPU_STK	Stats_Demo_Task_Stack[DEMO_TASK_STACK_SIZE];

/*
 * Function:	PrintDecU32
 *
 * Print a 32-bit unsigned decimal number.
 *
 * Globals:		None
 * 
 * Parameters:	value - the 32-bit number to print
 *
 * Return:		void
 */
void PrintDecU32( uint32 value )
{
	uint32 divisor = 1000000000;	/* Used to descend through the digits */
	
	uint32 num_to_print;			/* Digit in the 1s column */
	
	uint8 printing_started = 0;		/* Flag the first non-zero digit */

	/* Loop through the "tens" columns, most-to-least significant digit */
	while( divisor != 1 )
	{
		/* Pick out the value of the 1's column */
		num_to_print = value / divisor;
		num_to_print %= 10;
		
		/* Avoid printing leading zeros */
		if( num_to_print != 0 )
		{
			printing_started = 1;
		}
		
		/* Print the digit in the 1's column */
		if( printing_started )
		{
			LCD_PutChar( num_to_print + '0' );
		}
		
		divisor /= 10;
	} /* while( divisor != 1 ) */
	
	/* Handle the divisor == 1 case (also address value == 0 case ) */
	LCD_PutChar( ( value % 10 ) + '0' );
	
} /*PrintDecInt32() */


/*
 * Function:		Stats_Demo_Task
 *
 * This function runs as a uC/OS-III task with a medium priority. It loops
 * through a sequence of interesting statistics about the system.
 *
 * Execution of this task is controlled by its task semaphore. After
 * creating this task, the Main task starts the demo by posting to the
 * sempahore. When the user signals a desire to end the demo, it stops
 * the ADC and then pends on the semaphore. This leaves the task ready
 * to be re-executed at a later date and the ADC free for another task 
 * to use.
 *
 *			Start task
 *		 ->	Check the demo stop flag. If so...
 *		|		Pend on own task semaphore (sleep) 
 *      |   	Prepare the LCD and initalize the 'page' controlling variables
 *		|		Clear the stop flag
 *		|	Test if it is time to change the displayed stat. If so...
 *		|		Increment the stat to be shown and reset the timer
 *		|   Print the current stat info
 *		|	Delay the task so that update rate is not too fast
 *		|   Check if SW3 is pressed. If so...
 *		|   	Post to the main task semaphore (make it ready to run)
 *		|   	Set the stop flag
 *		 -- Loop
 *
 * Globals:		None
 *
 * Parameters:	p_arg (not used)
 *
 * Return:		None (infinite loop)
 */
void Stats_Demo_Task( void *p_arg )
{
	OS_ERR err;			/* Hold OS call return codes */
	
	uint32 value;		/* Hold system call return values */
	
	uint8 curr_stat;	/* The statistic being displayed */
	
	uint8 change_ctr;	/* Counter to time the page changes */
	
 	int stop_demo = 1;	/* Flag for SW3 press (and first time into the code) */
	
	(void)p_arg;		/* no-op prevents warning about unused p_arg */
   
	for( ; ; )		/* Demo main loop (forever) */
	{
		/* Execute when task first starts and when the demo is started */
		if( stop_demo )
		{
			/* Pend on own task semaphore so that MainTask can run */
			OSTaskSemPend(
				NO_SEM_TIMEOUT,
				OS_OPT_PEND_BLOCKING,
				NO_SEM_TIMESTAMP,
				&err );
			OSErrCheck( err, "Cannot post sem" );
	
			/*
			MainTask has posted to the TaskSem so it is time to run the demo.
			Prepare the LCD and initalize the 'page' controlling variables.
			*/
			LCD_ClearDisplay();
			LCD_PrintString( STATS_DEMO_TASK );
	
			curr_stat = STATS_VERSION;
			change_ctr = 0;
		
			/* Clear the flag so we only run this once per demo run */
			stop_demo = 0;
		
		} /* if( stop_demo ) */
		
		
		/* Increment counter and see if it is time change the display */
		change_ctr++;
		if( change_ctr == TIME_TO_CHANGE )
		{
			/* Changing display - increment the stat, reset the counter */
			change_ctr = 0;
			curr_stat++;
			curr_stat %= NUM_STATS;
		}
		
		/* Clear row 1 of the LCD */
		LCD_Position( 1, 0 );
		LCD_PrintString( SIXTEENSPACES );
		LCD_Position( 1, 0 );
		
		switch( curr_stat )
		{
			case STATS_VERSION:
			/* Print the OS version in format vXX.YY.ZZ */
				value = OSVersion( &err );

				LCD_PrintString( "uC/OS-III v" );
				LCD_PrintNumber( value / 10000 );
				LCD_PrintString( "." );
				LCD_PrintNumber( ( value / 100 ) % 100 );
				LCD_PrintString( "." );
				LCD_PrintNumber( value % 100 );	
			break;

			case STATS_TICK_RATE:
			/* Print the configured tick rate in Hertz */
				LCD_PrintString( "SYSTICK: " );
				PrintDecU32( OSCfg_TickRate_Hz );	
				LCD_PrintString( "Hz" );
			break;

			case STATS_CPU_USAGE:
			/* Print non-idle time as a percentage "xx.yy%" */
				LCD_PrintString( "CPU Usage: " );
				
				/* Note: OSStatTaskCPUUsage is a percentage * 100 */
				LCD_PrintNumber( OSStatTaskCPUUsage / 100 );
				
				LCD_PrintString( "." );
				
				/* Handle the RHS of the decimal point, with leading zero */
				value = OSStatTaskCPUUsage % 100;	
				if( value < 10 )
				{
					LCD_PrintString( "0" );
				}
				LCD_PrintNumber( value );
				LCD_PrintString( "%" );
			break;
			
			case STATS_CPU_FREQ:
			/* Print CPU clock speed in Mega-Hertz*/
				value = BSP_CPU_ClkFreq();

				LCD_PrintString( "CPU Speed: " );
				LCD_PrintNumber( value / 1000000 );
				LCD_PrintString( "MHz" );
			break;

			case STATS_TICKS:
			/* Print ticks count since OSStart */
				LCD_PrintString( "Ticks: " );
				PrintDecU32( OSTickCtr );

			break;

			case STATS_CTXSW:
			/* Print context switch count since OSStart */
				LCD_PrintString( "CtxSw: " );
				PrintDecU32( OSTaskCtxSwCtr );
			break;

			default:
				/* This should never happen */
				/* Just reset to the start for safety */
				curr_stat = 0;
				change_ctr = 0;
			break;
			
		} /* switch( curr_page ) */

		OSTimeDly(
			INFO_REFRESH_RATE,		
           	OS_OPT_TIME_DLY,
			&err );
		OSErrCheck( err, "Delay call failed" );
		
		/* Check to see if SW3 was pressed - this signals the end of the demo */
		if( ! P15_5_Read() )		/* Note: 0 (ground) means SW3 was pressed */
		{	
			/* Post to the MainTask semaphore to make it ready to run */
			OSTaskSemPost(
				&Main_Task_TCB,
				OS_OPT_POST_NONE,
				&err );
			OSErrCheck( err, "Cannot post to MainTask semaphore" );	
			
			/* Set flag to go to sleep (allow MainTask to run) */
			stop_demo = 1;
			
		} /* if( ! P15_5_Read() ) */
		
	} /* for( ; ; ) */
	
} /* Stats_Demo_Task */
